<?php
/* --------------------------------------------------------------
   ImageProcessingService.php 2023-01-30
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2023 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\ImageProcessing\Service;

use Gambio\Admin\Modules\ImageProcessing\Exceptions\ImageNotFoundException;
use Gambio\Admin\Modules\ImageProcessing\Exceptions\ImageProcessingFailedException;
use Gambio\Admin\Modules\ImageProcessing\Exceptions\UnsupportedImageTypeException;
use Gambio\Admin\Modules\ImageProcessing\Model\ImageProcessingSettings;

/**
 * Interface ImageProcessingService
 *
 * @package Gambio\Admin\Modules\ImageProcessing\Service
 */
interface ImageProcessingService
{
    /**
     * Creates a popup, info, thumbnail and gallery image.
     *
     * @param string $imageFile
     *
     * @throws ImageNotFoundException|UnsupportedImageTypeException|ImageProcessingFailedException
     */
    public function process(string $imageFile): void;
    
    
    /**
     * Creates a popup image from an original image.
     *
     * @param string $imageFile
     *
     * @throws ImageNotFoundException|UnsupportedImageTypeException|ImageProcessingFailedException
     */
    public function processPopup(string $imageFile): void;
    
    
    /**
     * Creates an info image from an original image.
     *
     * @param string $imageFile
     *
     * @throws ImageNotFoundException|UnsupportedImageTypeException|ImageProcessingFailedException
     */
    public function processInfoImage(string $imageFile): void;
    
    
    /**
     * Creates a thumbnail image from an original image.
     *
     * @param string $imageFile
     *
     * @throws ImageNotFoundException|UnsupportedImageTypeException|ImageProcessingFailedException
     */
    public function processThumbnail(string $imageFile): void;
    
    
    /**
     * Creates a gallery image from an original image.
     *
     * @param string $imageFile
     *
     * @throws ImageNotFoundException|UnsupportedImageTypeException|ImageProcessingFailedException
     */
    public function processGalleryImage(string $imageFile): void;
    
    
    /**
     * Creates a custom processed image.
     *
     * @param string                  $imageFile
     * @param ImageProcessingSettings $settings
     *
     * @throws ImageNotFoundException
     * @throws ImageProcessingFailedException
     * @throws UnsupportedImageTypeException
     */
    public function processImage(string $imageFile, ImageProcessingSettings $settings);
}